
  /*
   *  Object %name    : %
   *  State           :  %state%
   *  Creation date   :  Sun March 22 10:17:10 2009
   *  Last modified   :  %modify_time%
   */
  /** @file 
   *  \brief This file includes the KMNG activate and use RSA keys 
   *          ATP tests						
   *
   *  \version KMNG_SEP_ATP_RSA_V15_ActivateKeys.c#1:csrc:1
   *  \author Annag
   *  \remarks Copyright (C) 2007 by Discretix Technologies Ltd.
   *           All Rights reserved
   */

/************* Include Files ****************/
#include <stdio.h>
#include "DX_VOS_BaseTypes.h"
#include "DX_VOS_Mem.h"
#include "CRYS.h"
#include "CRYS_KMNG.h"
#include "CRYS_AES.h"
#include "CRYS_DES.h"
#include "CRYS_error.h"
#include "CRYS_HASH.h"
#include "CRYS_HMAC.h"
#include "CRYS_RSA_Types.h"
#include "KMNG_Defs.h"
#include "KMNG_API.h"
#include "KMNG_SEP_ATP_Defines.h"
#include "SEP_ATP_UTIL_funcs.h"
#include "KMNG_SEP_ATP_RSA_V15_ActivateKeys.h"

#include "KMNG_Defs.h"
#include "LLF_KMNG.h"
#include "KMNG_SEP_ATP_ImportExportKeys.h"
#include "KMNG_SEP_ATP_TestDataDefines.h"



/************************ Defines *******************************/
/************************ Enums *********************************/
/************************ Typedefs ******************************/
/************************ Global Data ***************************/

/* pointer to the struct with RSA key test data vector */
extern TST_Ext_KeysRSA_TestDataStruct *SEP_KeysOpRSA_TestData;




/************* Private function prototype ***********************/

/***********************************************************************
* Function Name:                                                           
*  SEP_KMNG_ATP_Ext_RSA_PKCS1V15_EncDec_ActivateOperation         
* 
* Inputs:
*
* Outputs:
*  CRYSError_t - Function error return                                     
*
* Description:                                                                 
*  This function Is a Acceptance test for RSA key activate operation.
* Algorithm:       
* 1. Create a master key
* 2. Import RSA key
* 3. Activate RSA key for encrypt operation
* 4. Execute RSA PKCS1V15_Encrypt operation with activated key
* 5. Activate RSA key for decrypt operation
* 6. Execute RSA PKCS1V15_Decrypt operation with activated key
* 7. Verify that decrypted message is as original message
**************************************************************************************/                                
DxError_t SEP_KMNG_ATP_Ext_RSA_PKCS1V15_EncDec_ActivateOperation( void ) 
{

	/********* TST Variables Definitions ************/
	DxUint32_t				TST_index = 0; 
	DxError_t				TST_Status = DX_SUCCESS; 
	DxUint8_t				TST_KeyRingPwd[TST_KMNG_PWD_MAX_LEN] = {0x0,0x1,0x2,0x3,0x4,0x5,0x6,0x7,0x8,0x9};
	DxUint32_t				TST_KeyRingPwdLen = 10;
	DxChar					TST_MasterTestName[TST_FUNC_STRING_LEN] = "SEP_KMNG_ATP_Ext_RSA_PKCS1V15_EncDec_ActivateOperation";     
	KMNG_RSA_WrappedKey_t	TST_RSAWrappedKey;
	KMNG_RSA_WrappedKey_t	TST_SavedRSAWrappedKey;

	/* paramaters for create key */
	KMNG_KeyDataParameter_t		TST_masterKeyDataParam;

	/* configuration parameters */
	KMNG_KeyConfiguration_t		TST_masterKeyConfiguration;

	/* paramaters for create key */
	KMNG_KeyDataParameter_t		TST_userKeyDataParam;

	/* configuration parameters */
	KMNG_KeyConfiguration_t		TST_userKeyConfiguration;

	// for master key
	DxUint8_t					TST_encryptedKeyBuffer[KMNG_SYM_ENCR_KEY_BUFF_SIZE_IN_BYTES];
	DxUint8_t					TST_userEncryptedKeyBuffer[KMNG_RSA_ENCR_KEY_BUFF_SIZE_IN_BYTES];

	/* buffer to save encrypted key */
	DxUint8_t					TST_savedEncryptedKeyBuffer[KMNG_SYM_ENCR_KEY_BUFF_SIZE_IN_BYTES];
	DxUint32_t					TST_savedEncryptedKeyDataLength;

	DxUint8_t					TST_savedUserEncryptedKeyBuffer[KMNG_RSA_ENCR_KEY_BUFF_SIZE_IN_BYTES];
	DxUint32_t					TST_savedUserEncryptedKeyDataLength;
	/* key size  */
	DxUint32_t                  TST_KeySize;
	CRYS_RSAPrimeData_t			TST_PrimeData;
	DxUint8_t                   TST_Output[TST_KMNG_MAX_SIZE_DATA] = {0};
	DxUint16_t				    TST_OutputSize = TST_KMNG_MAX_SIZE_DATA;
	DxUint16_t                  TST_DataSize;

	TST_DataSize = SEP_KeysOpRSA_TestData[TST_index].TST_DataLen - TST_KMNG_LESS_DATA_FOR_PKCS1V15_ENCRYPT;


	/* set master key parameters (should be AES key) */
	TST_masterKeyDataParam.keyPassword_ptr = TST_KeyRingPwd;
	TST_masterKeyDataParam.keyPasswordLen  = TST_KeyRingPwdLen;
	TST_masterKeyDataParam.encryptedKeyData_ptr   = (DxUint8_t*)TST_encryptedKeyBuffer;
	TST_masterKeyDataParam.encryptedKeyDataLength = KMNG_SYM_ENCR_KEY_BUFF_SIZE_IN_BYTES;

	/* set master key configuration */
	TST_masterKeyConfiguration.type = KMNG_KeyTypeAES; 
	TST_masterKeyConfiguration.keySizeBytes = KMNG_AES_MAX_KEY_SIZE_IN_BYTES; /*256 Bits Key*/
	TST_masterKeyConfiguration.usage = KMNG_KEY_USAGE_STORAGE;
	TST_masterKeyConfiguration.restriction = KMNG_KEY_RESTRICTION_NON_EXPORTABLE;  /* instead PROTECTED.  May be other?? */

	/* 1. Create a master key */
	/**************************/
	TST_Status = LLF_KMNG_CreateKeyRingKey(&TST_masterKeyDataParam , &TST_masterKeyConfiguration);

	TST_ATP_CHECK_ERROR_RETURN(TST_Status,
		"LLF_KMNG_CreateKeyRingKey",
		SEP_KeysOpRSA_TestData[TST_index].TST_Name,
		TST_MasterTestName);


	/* save master key, because it will be unwrapped in each KMNG operation */
	DX_VOS_FastMemCpy(TST_savedEncryptedKeyBuffer, TST_masterKeyDataParam.encryptedKeyData_ptr, KMNG_SYM_ENCR_KEY_BUFF_SIZE_IN_BYTES);
	TST_savedEncryptedKeyDataLength = TST_masterKeyDataParam.encryptedKeyDataLength;

	/* set user key paramaters */
	TST_userKeyDataParam.keyPassword_ptr = SEP_KeysOpRSA_TestData[TST_index].TST_KeyPwd;
	TST_userKeyDataParam.keyPasswordLen = SEP_KeysOpRSA_TestData[TST_index].TST_KeyPwdLen;
	/* TST_userKeyDataParam.keyId - will return from LLF layer */
	TST_userKeyDataParam.encryptedKeyData_ptr = (DxUint8_t*)TST_userEncryptedKeyBuffer;
	TST_userKeyDataParam.encryptedKeyDataLength = KMNG_RSA_ENCR_KEY_BUFF_SIZE_IN_BYTES;


	/* set key configuration */
	TST_userKeyConfiguration.type         = SEP_KeysOpRSA_TestData[TST_index].TST_KeyType;
	TST_userKeyConfiguration.usage        = SEP_KeysOpRSA_TestData[TST_index].TST_KeyUsage;
	TST_userKeyConfiguration.keySizeBytes = SEP_KeysOpRSA_TestData[TST_index].TST_KeySize;  
	TST_userKeyConfiguration.restriction  = SEP_KeysOpRSA_TestData[TST_index].TST_KeyRestriction;

	/* if UserSpecificKeyData is not NULL, then copy user specific data into configuration structure,
	else set it to 0  */
	TST_userKeyConfiguration.UserSpecificKeyData[0] = SEP_KeysOpRSA_TestData[TST_index].TST_UserData[0];
	TST_userKeyConfiguration.UserSpecificKeyData[1] = SEP_KeysOpRSA_TestData[TST_index].TST_UserData[1];  	

	/* 2. Import RSA key */
	/*********************/
	TST_Status = LLF_KMNG_ImportRSAUserKey(&TST_masterKeyDataParam , 
		&TST_userKeyConfiguration , 
		&TST_userKeyDataParam , 
		SEP_KeysOpRSA_TestData[TST_index].TST_N,
		SEP_KeysOpRSA_TestData[TST_index].TST_NSize,
		SEP_KeysOpRSA_TestData[TST_index].TST_E,
		SEP_KeysOpRSA_TestData[TST_index].TST_ESize,
		SEP_KeysOpRSA_TestData[TST_index].TST_D,
		SEP_KeysOpRSA_TestData[TST_index].TST_DSize);

	TST_ATP_CHECK_ERROR_RETURN( TST_Status,
		"LLF_KMNG_ImportRSAUserKey",
		SEP_KeysOpRSA_TestData[TST_index].TST_Name,
		TST_MasterTestName);

	/* save user key, because it will be unwrapped in each KMNG operation */
	DX_VOS_FastMemCpy(TST_savedUserEncryptedKeyBuffer, TST_userKeyDataParam.encryptedKeyData_ptr, KMNG_RSA_ENCR_KEY_BUFF_SIZE_IN_BYTES);
	TST_savedUserEncryptedKeyDataLength = TST_userKeyDataParam.encryptedKeyDataLength;

	/* restore the encrypted key: */
	DX_VOS_FastMemCpy(TST_masterKeyDataParam.encryptedKeyData_ptr, TST_savedEncryptedKeyBuffer, KMNG_SYM_ENCR_KEY_BUFF_SIZE_IN_BYTES);
	TST_masterKeyDataParam.encryptedKeyDataLength = TST_savedEncryptedKeyDataLength;

	/* 3. Activate RSA key with type KMNG_RsaKeyOperationEncrypt */
	/*************************************************************/
	TST_KeySize  = sizeof(KMNG_RSA_WrappedKey_t);
	TST_Status = LLF_KMNG_ActivateRSAKey(&TST_masterKeyDataParam,
		&TST_userKeyDataParam,
		KMNG_RsaKeyOperationEncrypt,
		(DxUint8_t*)TST_RSAWrappedKey,
		&TST_KeySize);


	TST_ATP_CHECK_ERROR_RETURN(TST_Status,
		"LLF_KMNG_ActivateRSAKey",
		SEP_KeysOpRSA_TestData[TST_index].TST_Name,
		TST_MasterTestName);

	/* restore the encrypted key: */
	DX_VOS_FastMemCpy(TST_masterKeyDataParam.encryptedKeyData_ptr, TST_savedEncryptedKeyBuffer, KMNG_SYM_ENCR_KEY_BUFF_SIZE_IN_BYTES);
	TST_masterKeyDataParam.encryptedKeyDataLength = TST_savedEncryptedKeyDataLength;

	/* save RSA wrapped key */
	DX_VOS_FastMemCpy(TST_SavedRSAWrappedKey, TST_RSAWrappedKey, KMNG_RSA_WRAP_KEY_LEN);

	/* 4. Execute RSA PRIM_Encrypt operation with activated key */
	/************************************************************/
	TST_Status = CRYS_KMNG_RSA_PKCS1v15_Encrypt(TST_RSAWrappedKey,
		&TST_PrimeData,
		SEP_KeysOpRSA_TestData[TST_index].TST_Data,
		TST_DataSize,
		TST_Output);


	TST_ATP_CHECK_ERROR_RETURN(TST_Status,
		"CRYS_KMNG_RSA_PKCS1v15_Encrypt",
		SEP_KeysOpRSA_TestData[TST_index].TST_Name,
		TST_MasterTestName);


	/* restore the encrypted key: */
	DX_VOS_FastMemCpy(TST_userKeyDataParam.encryptedKeyData_ptr, TST_savedUserEncryptedKeyBuffer, KMNG_RSA_ENCR_KEY_BUFF_SIZE_IN_BYTES);
	TST_userKeyDataParam.encryptedKeyDataLength = TST_savedUserEncryptedKeyDataLength;

	TST_KeySize  = sizeof(KMNG_RSA_WrappedKey_t);

	/* 5. Activate RSA key with type KMNG_RsaKeyOperationDecrypt */
	/*************************************************************/
	TST_Status = LLF_KMNG_ActivateRSAKey(&TST_masterKeyDataParam,
		&TST_userKeyDataParam,
		KMNG_RsaKeyOperationDecrypt,
		(DxUint8_t*)TST_RSAWrappedKey,
		&TST_KeySize);

	TST_ATP_CHECK_ERROR_RETURN(TST_Status,
		"LLF_KMNG_ActivateRSAKey",
		SEP_KeysOpRSA_TestData[TST_index].TST_Name,
		TST_MasterTestName);

	/* 6. Execute RSA PKCS1v15_Decrypt operation with activated key */
	/************************************************************/
	TST_Status = CRYS_KMNG_RSA_PKCS1v15_Decrypt(TST_RSAWrappedKey,
		&TST_PrimeData,	
		TST_Output,
		SEP_KeysOpRSA_TestData[TST_index].TST_DataLen,
		TST_Output,
		&TST_OutputSize);


	TST_ATP_CHECK_ERROR_RETURN(TST_Status,
		"CRYS_KMNG_RSA_PKCS1v15_Decrypt",
		SEP_KeysOpRSA_TestData[TST_index].TST_Name,
		TST_MasterTestName);

	/* 7. Verify that decrypted message is as original message */
	/***********************************************************/
	TST_Status = (DxUint32_t)DX_VOS_MemCmp(SEP_KeysOpRSA_TestData[TST_index].TST_Data,TST_Output,TST_OutputSize);

	TST_ATP_CHECK_ERROR_RETURN( TST_Status,
		"DX_VOS_MemCmp",
		SEP_KeysOpRSA_TestData[TST_index].TST_Name,
		TST_MasterTestName);

EXIT_ON_ERROR:
	return TST_Status; 

}

/***********************************************************************
* Function Name:                                                           
*  SEP_KMNG_ATP_Ext_RSA_V15_SignVerify_Integrated_ActivateOperation         
* 
* Inputs:
*
* Outputs:
*  CRYSError_t - Function error return                                     
*
* Description:                                                                 
*  This function Is a Acceptance test for RSA key activate operation.
* Algorithm:       
* 1. Create a master key
* 2. Import RSA key
* 3. Activate RSA key for decrypt operation
* 4. Execute RSA CRYS_KMNG_RSA_PKCS1v15_Sign operation with activated key
* 5. Activate RSA key for encrypt operation
* 6. Execute CRYS_KMNG_RSA_PKCS1v15_Verify operation with activated key
**************************************************************************************/                                
DxError_t SEP_KMNG_ATP_Ext_RSA_V15_SignVerify_Integrated_ActivateOperation( void ) 
{

	/********* TST Variables Definitions ************/
	DxUint32_t				TST_index = 0; 
	DxError_t				TST_Status = DX_SUCCESS; 
	DxUint8_t				TST_KeyRingPwd[TST_KMNG_PWD_MAX_LEN] = {0x0,0x1,0x2,0x3,0x4,0x5,0x6,0x7,0x8,0x9};
	DxUint32_t				TST_KeyRingPwdLen = 10;
	DxChar					TST_MasterTestName[TST_FUNC_STRING_LEN] = "SEP_KMNG_ATP_Ext_RSA_V15_SignVerify_Integrated_ActivateOperation";     
	KMNG_RSA_WrappedKey_t	TST_RSAWrappedKey;
	KMNG_RSA_WrappedKey_t	TST_SavedRSAWrappedKey;

	/* paramaters for create key */
	KMNG_KeyDataParameter_t		TST_masterKeyDataParam;

	/* configuration parameters */
	KMNG_KeyConfiguration_t		TST_masterKeyConfiguration;

	/* paramaters for create key */
	KMNG_KeyDataParameter_t		TST_userKeyDataParam;

	/* configuration parameters */
	KMNG_KeyConfiguration_t		TST_userKeyConfiguration;

	// for master key
	DxUint8_t					TST_encryptedKeyBuffer[KMNG_SYM_ENCR_KEY_BUFF_SIZE_IN_BYTES];
	DxUint8_t					TST_userEncryptedKeyBuffer[KMNG_RSA_ENCR_KEY_BUFF_SIZE_IN_BYTES];

	/* buffer to save encrypted key */
	DxUint8_t					TST_savedEncryptedKeyBuffer[KMNG_SYM_ENCR_KEY_BUFF_SIZE_IN_BYTES];
	DxUint32_t					TST_savedEncryptedKeyDataLength;

	DxUint8_t					TST_savedUserEncryptedKeyBuffer[KMNG_RSA_ENCR_KEY_BUFF_SIZE_IN_BYTES];
	DxUint32_t					TST_savedUserEncryptedKeyDataLength;
	/* key size  */
	DxUint32_t                  TST_KeySize;
	CRYS_RSAPrivUserContext_t   TST_PrivContext;
	CRYS_RSAPubUserContext_t    TST_PubContext;
	DxUint8_t                   TST_Output[TST_KMNG_MAX_SIZE_DATA] = {0};
	DxUint16_t				    TST_OutputSize = TST_KMNG_MAX_SIZE_DATA;


	/* set master key parameters (should be AES key) */
	TST_masterKeyDataParam.keyPassword_ptr = TST_KeyRingPwd;
	TST_masterKeyDataParam.keyPasswordLen  = TST_KeyRingPwdLen;
	TST_masterKeyDataParam.encryptedKeyData_ptr   = (DxUint8_t*)TST_encryptedKeyBuffer;
	TST_masterKeyDataParam.encryptedKeyDataLength = KMNG_SYM_ENCR_KEY_BUFF_SIZE_IN_BYTES;

	/* set master key configuration */
	TST_masterKeyConfiguration.type = KMNG_KeyTypeAES; 
	TST_masterKeyConfiguration.keySizeBytes = KMNG_AES_MAX_KEY_SIZE_IN_BYTES; /*256 Bits Key*/
	TST_masterKeyConfiguration.usage = KMNG_KEY_USAGE_STORAGE;
	TST_masterKeyConfiguration.restriction = KMNG_KEY_RESTRICTION_NON_EXPORTABLE;  /* instead PROTECTED.  May be other?? */

	/* 1. Create a master key */
	/**************************/
	TST_Status = LLF_KMNG_CreateKeyRingKey(&TST_masterKeyDataParam , &TST_masterKeyConfiguration);

	TST_ATP_CHECK_ERROR_RETURN(TST_Status,
		"LLF_KMNG_CreateKeyRingKey",
		SEP_KeysOpRSA_TestData[TST_index].TST_Name,
		TST_MasterTestName);


	/* save master key, because it will be unwrapped in each KMNG operation */
	DX_VOS_FastMemCpy(TST_savedEncryptedKeyBuffer, TST_masterKeyDataParam.encryptedKeyData_ptr, KMNG_SYM_ENCR_KEY_BUFF_SIZE_IN_BYTES);
	TST_savedEncryptedKeyDataLength = TST_masterKeyDataParam.encryptedKeyDataLength;

	/* set user key parameters */
	TST_userKeyDataParam.keyPassword_ptr = SEP_KeysOpRSA_TestData[TST_index].TST_KeyPwd;
	TST_userKeyDataParam.keyPasswordLen = SEP_KeysOpRSA_TestData[TST_index].TST_KeyPwdLen;
	/* TST_userKeyDataParam.keyId - will return from LLF layer */
	TST_userKeyDataParam.encryptedKeyData_ptr = (DxUint8_t*)TST_userEncryptedKeyBuffer;
	TST_userKeyDataParam.encryptedKeyDataLength = KMNG_RSA_ENCR_KEY_BUFF_SIZE_IN_BYTES;


	/* set key configuration */
	TST_userKeyConfiguration.type         = SEP_KeysOpRSA_TestData[TST_index].TST_KeyType;
	TST_userKeyConfiguration.usage        = SEP_KeysOpRSA_TestData[TST_index].TST_KeyUsage;
	TST_userKeyConfiguration.keySizeBytes = SEP_KeysOpRSA_TestData[TST_index].TST_KeySize;  
	TST_userKeyConfiguration.restriction  = SEP_KeysOpRSA_TestData[TST_index].TST_KeyRestriction;

	/* if UserSpecificKeyData is not NULL, then copy user specific data into configuration structure,
	else set it to 0  */
	TST_userKeyConfiguration.UserSpecificKeyData[0] = SEP_KeysOpRSA_TestData[TST_index].TST_UserData[0];
	TST_userKeyConfiguration.UserSpecificKeyData[1] = SEP_KeysOpRSA_TestData[TST_index].TST_UserData[1];  	

	/* 2. Import RSA key */
	/*********************/
	TST_Status = LLF_KMNG_ImportRSAUserKey(&TST_masterKeyDataParam , 
		&TST_userKeyConfiguration , 
		&TST_userKeyDataParam , 
		SEP_KeysOpRSA_TestData[TST_index].TST_N,
		SEP_KeysOpRSA_TestData[TST_index].TST_NSize,
		SEP_KeysOpRSA_TestData[TST_index].TST_E,
		SEP_KeysOpRSA_TestData[TST_index].TST_ESize,
		SEP_KeysOpRSA_TestData[TST_index].TST_D,
		SEP_KeysOpRSA_TestData[TST_index].TST_DSize);

	TST_ATP_CHECK_ERROR_RETURN( TST_Status,
		"LLF_KMNG_ImportRSAUserKey",
		SEP_KeysOpRSA_TestData[TST_index].TST_Name,
		TST_MasterTestName);

	/* save user key, because it will be unwrapped in each KMNG operation */
	DX_VOS_FastMemCpy(TST_savedUserEncryptedKeyBuffer, TST_userKeyDataParam.encryptedKeyData_ptr, KMNG_RSA_ENCR_KEY_BUFF_SIZE_IN_BYTES);
	TST_savedUserEncryptedKeyDataLength = TST_userKeyDataParam.encryptedKeyDataLength;

	/* restore the encrypted key: */
	DX_VOS_FastMemCpy(TST_masterKeyDataParam.encryptedKeyData_ptr, TST_savedEncryptedKeyBuffer, KMNG_SYM_ENCR_KEY_BUFF_SIZE_IN_BYTES);
	TST_masterKeyDataParam.encryptedKeyDataLength = TST_savedEncryptedKeyDataLength;

	/* 3. Activate RSA key with type KMNG_RsaKeyOperationEncrypt */
	/*************************************************************/
	TST_KeySize  = sizeof(KMNG_RSA_WrappedKey_t);
	TST_Status = LLF_KMNG_ActivateRSAKey(&TST_masterKeyDataParam,
		&TST_userKeyDataParam,
		KMNG_RsaKeyOperationDecrypt,
		(DxUint8_t*)TST_RSAWrappedKey,
		&TST_KeySize);


	TST_ATP_CHECK_ERROR_RETURN(TST_Status,
		"LLF_KMNG_ActivateRSAKey",
		SEP_KeysOpRSA_TestData[TST_index].TST_Name,
		TST_MasterTestName);

	/* restore the encrypted key: */
	DX_VOS_FastMemCpy(TST_masterKeyDataParam.encryptedKeyData_ptr, TST_savedEncryptedKeyBuffer, KMNG_SYM_ENCR_KEY_BUFF_SIZE_IN_BYTES);
	TST_masterKeyDataParam.encryptedKeyDataLength = TST_savedEncryptedKeyDataLength;

	/* save RSA wrapped key */
	DX_VOS_FastMemCpy(TST_SavedRSAWrappedKey, TST_RSAWrappedKey, KMNG_RSA_WRAP_KEY_LEN);

	/* 4. Execute RSA CRYS_KMNG_RSA_PKCS1v15_Sign operation with activated key */
	/***************************************************************************/
	TST_Status = CRYS_KMNG_RSA_PKCS1v15_Sign(TST_RSAWrappedKey,
		&TST_PrivContext,
		SEP_KeysOpRSA_TestData[TST_index].TST_HASHMode,
		SEP_KeysOpRSA_TestData[TST_index].TST_Data,
		SEP_KeysOpRSA_TestData[TST_index].TST_DataLen,
		TST_Output,
		&TST_OutputSize);

	TST_ATP_CHECK_ERROR_RETURN(TST_Status,
		"CRYS_KMNG_RSA_PKCS1v15_Sign",
		SEP_KeysOpRSA_TestData[TST_index].TST_Name,
		TST_MasterTestName);


	/* restore the encrypted key: */
	DX_VOS_FastMemCpy(TST_userKeyDataParam.encryptedKeyData_ptr, TST_savedUserEncryptedKeyBuffer, KMNG_RSA_ENCR_KEY_BUFF_SIZE_IN_BYTES);
	TST_userKeyDataParam.encryptedKeyDataLength = TST_savedUserEncryptedKeyDataLength;

	TST_KeySize  = sizeof(KMNG_RSA_WrappedKey_t);

	/* 5. Activate RSA key with type KMNG_RsaKeyOperationDecrypt */
	/*************************************************************/
	TST_Status = LLF_KMNG_ActivateRSAKey(&TST_masterKeyDataParam,
		&TST_userKeyDataParam,
		KMNG_RsaKeyOperationEncrypt,
		(DxUint8_t*)TST_RSAWrappedKey,
		&TST_KeySize);

	TST_ATP_CHECK_ERROR_RETURN(TST_Status,
		"LLF_KMNG_ActivateRSAKey",
		SEP_KeysOpRSA_TestData[TST_index].TST_Name,
		TST_MasterTestName);

	/* 6. Execute CRYS_KMNG_RSA_PKCS1v15_Verify operation with activated key */
	/*************************************************************************/
	TST_Status = CRYS_KMNG_RSA_PKCS1v15_Verify(TST_RSAWrappedKey,
		&TST_PubContext,
		SEP_KeysOpRSA_TestData[TST_index].TST_HASHMode,
		SEP_KeysOpRSA_TestData[TST_index].TST_Data,
		SEP_KeysOpRSA_TestData[TST_index].TST_DataLen,
		TST_Output);

	TST_ATP_CHECK_ERROR_RETURN(TST_Status,
		"CRYS_KMNG_RSA_PKCS1v15_Verify",
		SEP_KeysOpRSA_TestData[TST_index].TST_Name,
		TST_MasterTestName);

EXIT_ON_ERROR:
	return TST_Status; 

}

/***********************************************************************
* Function Name:                                                           
*  SEP_KMNG_ATP_Ext_RSA_V15_SignVerify_NonIntegrated_ActivateOperation         
* 
* Inputs:
*
* Outputs:
*  CRYSError_t - Function error return                                     
*
* Description:                                                                 
*  This function Is a Acceptance test for RSA key activate operation.
* Algorithm:       
* 1. Create a master key
* 2. Import RSA key
* 3. Activate RSA key for decrypt operation
* 4. Execute RSA CRYS_KMNG_RSA_PKCS1v15_SignInit operation with activated key
* 5. Execute RSA CRYS_RSA_PKCS1v15_SignUpdate operation with activated key
* 6. Execute RSA CRYS_RSA_PKCS1v15_SignFinish operation with activated key
* 7. Activate RSA key for encrypt operation
* 8. Execute CRYS_KMNG_RSA_PKCS1v15_VerifyInit operation with activated key
* 9. Execute CRYS_KMNG_RSA_PKCS1v15_VerifyUpdate operation with activated key
* 10. Execute CRYS_KMNG_RSA_PKCS1v15_VerifyFinish operation with activated key
**************************************************************************************/                                
DxError_t SEP_KMNG_ATP_Ext_RSA_V15_SignVerify_NonIntegrated_ActivateOperation( void ) 
{

	/********* TST Variables Definitions ************/
	DxUint32_t				TST_index = 0; 
	DxError_t				TST_Status = DX_SUCCESS; 
	DxUint8_t				TST_KeyRingPwd[TST_KMNG_PWD_MAX_LEN] = {0x0,0x1,0x2,0x3,0x4,0x5,0x6,0x7,0x8,0x9};
	DxUint32_t				TST_KeyRingPwdLen = 10;
	DxChar					TST_MasterTestName[TST_FUNC_STRING_LEN] = "SEP_KMNG_ATP_Ext_RSA_V15_SignVerify_NonIntegrated_ActivateOperation";     
	KMNG_RSA_WrappedKey_t	TST_RSAWrappedKey;
	KMNG_RSA_WrappedKey_t	TST_SavedRSAWrappedKey;

	/* paramaters for create key */
	KMNG_KeyDataParameter_t		TST_masterKeyDataParam;

	/* configuration parameters */
	KMNG_KeyConfiguration_t		TST_masterKeyConfiguration;

	/* paramaters for create key */
	KMNG_KeyDataParameter_t		TST_userKeyDataParam;

	/* configuration parameters */
	KMNG_KeyConfiguration_t		TST_userKeyConfiguration;

	// for master key
	DxUint8_t					TST_encryptedKeyBuffer[KMNG_SYM_ENCR_KEY_BUFF_SIZE_IN_BYTES];
	DxUint8_t					TST_userEncryptedKeyBuffer[KMNG_RSA_ENCR_KEY_BUFF_SIZE_IN_BYTES];

	/* buffer to save encrypted key */
	DxUint8_t					TST_savedEncryptedKeyBuffer[KMNG_SYM_ENCR_KEY_BUFF_SIZE_IN_BYTES];
	DxUint32_t					TST_savedEncryptedKeyDataLength;

	DxUint8_t					TST_savedUserEncryptedKeyBuffer[KMNG_RSA_ENCR_KEY_BUFF_SIZE_IN_BYTES];
	DxUint32_t					TST_savedUserEncryptedKeyDataLength;
	/* key size  */
	DxUint32_t                  TST_KeySize;
	CRYS_RSAPrivUserContext_t   TST_PrivContext;
	CRYS_RSAPubUserContext_t    TST_PubContext;
	DxUint8_t                   TST_Output[TST_KMNG_MAX_SIZE_DATA] = {0};
	DxUint16_t				    TST_OutputSize = TST_KMNG_MAX_SIZE_DATA;


	/* set master key parameters (should be AES key) */
	TST_masterKeyDataParam.keyPassword_ptr = TST_KeyRingPwd;
	TST_masterKeyDataParam.keyPasswordLen  = TST_KeyRingPwdLen;
	TST_masterKeyDataParam.encryptedKeyData_ptr   = (DxUint8_t*)TST_encryptedKeyBuffer;
	TST_masterKeyDataParam.encryptedKeyDataLength = KMNG_SYM_ENCR_KEY_BUFF_SIZE_IN_BYTES;

	/* set master key configuration */
	TST_masterKeyConfiguration.type = KMNG_KeyTypeAES; 
	TST_masterKeyConfiguration.keySizeBytes = KMNG_AES_MAX_KEY_SIZE_IN_BYTES; /*256 Bits Key*/
	TST_masterKeyConfiguration.usage = KMNG_KEY_USAGE_STORAGE;
	TST_masterKeyConfiguration.restriction = KMNG_KEY_RESTRICTION_NON_EXPORTABLE;  /* instead PROTECTED.  May be other?? */

	/* 1. Create a master key */
	/**************************/
	TST_Status = LLF_KMNG_CreateKeyRingKey(&TST_masterKeyDataParam , &TST_masterKeyConfiguration);

	TST_ATP_CHECK_ERROR_RETURN(TST_Status,
		"LLF_KMNG_CreateKeyRingKey",
		SEP_KeysOpRSA_TestData[TST_index].TST_Name,
		TST_MasterTestName);


	/* save master key, because it will be unwrapped in each KMNG operation */
	DX_VOS_FastMemCpy(TST_savedEncryptedKeyBuffer, TST_masterKeyDataParam.encryptedKeyData_ptr, KMNG_SYM_ENCR_KEY_BUFF_SIZE_IN_BYTES);
	TST_savedEncryptedKeyDataLength = TST_masterKeyDataParam.encryptedKeyDataLength;

	/* set user key parameters */
	TST_userKeyDataParam.keyPassword_ptr = SEP_KeysOpRSA_TestData[TST_index].TST_KeyPwd;
	TST_userKeyDataParam.keyPasswordLen = SEP_KeysOpRSA_TestData[TST_index].TST_KeyPwdLen;
	/* TST_userKeyDataParam.keyId - will return from LLF layer */
	TST_userKeyDataParam.encryptedKeyData_ptr = (DxUint8_t*)TST_userEncryptedKeyBuffer;
	TST_userKeyDataParam.encryptedKeyDataLength = KMNG_RSA_ENCR_KEY_BUFF_SIZE_IN_BYTES;


	/* set key configuration */
	TST_userKeyConfiguration.type         = SEP_KeysOpRSA_TestData[TST_index].TST_KeyType;
	TST_userKeyConfiguration.usage        = SEP_KeysOpRSA_TestData[TST_index].TST_KeyUsage;
	TST_userKeyConfiguration.keySizeBytes = SEP_KeysOpRSA_TestData[TST_index].TST_KeySize;  
	TST_userKeyConfiguration.restriction  = SEP_KeysOpRSA_TestData[TST_index].TST_KeyRestriction;

	/* if UserSpecificKeyData is not NULL, then copy user specific data into configuration structure,
	else set it to 0  */
	TST_userKeyConfiguration.UserSpecificKeyData[0] = SEP_KeysOpRSA_TestData[TST_index].TST_UserData[0];
	TST_userKeyConfiguration.UserSpecificKeyData[1] = SEP_KeysOpRSA_TestData[TST_index].TST_UserData[1];  	

	/* 2. Import RSA key */
	/*********************/
	TST_Status = LLF_KMNG_ImportRSAUserKey(&TST_masterKeyDataParam , 
		&TST_userKeyConfiguration , 
		&TST_userKeyDataParam , 
		SEP_KeysOpRSA_TestData[TST_index].TST_N,
		SEP_KeysOpRSA_TestData[TST_index].TST_NSize,
		SEP_KeysOpRSA_TestData[TST_index].TST_E,
		SEP_KeysOpRSA_TestData[TST_index].TST_ESize,
		SEP_KeysOpRSA_TestData[TST_index].TST_D,
		SEP_KeysOpRSA_TestData[TST_index].TST_DSize);

	TST_ATP_CHECK_ERROR_RETURN( TST_Status,
		"LLF_KMNG_ImportRSAUserKey",
		SEP_KeysOpRSA_TestData[TST_index].TST_Name,
		TST_MasterTestName);

	/* save user key, because it will be unwrapped in each KMNG operation */
	DX_VOS_FastMemCpy(TST_savedUserEncryptedKeyBuffer, TST_userKeyDataParam.encryptedKeyData_ptr, KMNG_RSA_ENCR_KEY_BUFF_SIZE_IN_BYTES);
	TST_savedUserEncryptedKeyDataLength = TST_userKeyDataParam.encryptedKeyDataLength;

	/* restore the encrypted key: */
	DX_VOS_FastMemCpy(TST_masterKeyDataParam.encryptedKeyData_ptr, TST_savedEncryptedKeyBuffer, KMNG_SYM_ENCR_KEY_BUFF_SIZE_IN_BYTES);
	TST_masterKeyDataParam.encryptedKeyDataLength = TST_savedEncryptedKeyDataLength;

	/* 3. Activate RSA key with type KMNG_RsaKeyOperationEncrypt */
	/*************************************************************/
	TST_KeySize  = sizeof(KMNG_RSA_WrappedKey_t);
	TST_Status = LLF_KMNG_ActivateRSAKey(&TST_masterKeyDataParam,
		&TST_userKeyDataParam,
		KMNG_RsaKeyOperationDecrypt,
		(DxUint8_t*)TST_RSAWrappedKey,
		&TST_KeySize);


	TST_ATP_CHECK_ERROR_RETURN(TST_Status,
		"LLF_KMNG_ActivateRSAKey",
		SEP_KeysOpRSA_TestData[TST_index].TST_Name,
		TST_MasterTestName);

	/* restore the encrypted key: */
	DX_VOS_FastMemCpy(TST_masterKeyDataParam.encryptedKeyData_ptr, TST_savedEncryptedKeyBuffer, KMNG_SYM_ENCR_KEY_BUFF_SIZE_IN_BYTES);
	TST_masterKeyDataParam.encryptedKeyDataLength = TST_savedEncryptedKeyDataLength;

	/* save RSA wrapped key */
	DX_VOS_FastMemCpy(TST_SavedRSAWrappedKey, TST_RSAWrappedKey, KMNG_RSA_WRAP_KEY_LEN);

	/* 4. Execute RSA CRYS_KMNG_RSA_PKCS1v15_SignInit operation with activated key */
	/************************************************************/
	TST_Status = CRYS_KMNG_RSA_PKCS1v15_SignInit(TST_RSAWrappedKey,
		&TST_PrivContext,
		SEP_KeysOpRSA_TestData[TST_index].TST_HASHMode);

	TST_ATP_CHECK_ERROR_RETURN(TST_Status,
		"CRYS_KMNG_RSA_PKCS1v15_SignInit",
		SEP_KeysOpRSA_TestData[TST_index].TST_Name,
		TST_MasterTestName);

	/* 5. Execute RSA CRYS_RSA_PKCS1v15_SignUpdate operation with activated key */
	/************************************************************/
	TST_Status = CRYS_RSA_PKCS1v15_SignUpdate(
		&TST_PrivContext,
		SEP_KeysOpRSA_TestData[TST_index].TST_Data,
		SEP_KeysOpRSA_TestData[TST_index].TST_DataLen);

	TST_ATP_CHECK_ERROR_RETURN(TST_Status,
		"CRYS_RSA_PKCS1v15_SignUpdate",
		SEP_KeysOpRSA_TestData[TST_index].TST_Name,
		TST_MasterTestName);

	/* 6. Execute RSA CRYS_RSA_PKCS1v15_SignFinish operation with activated key */
	/************************************************************/
	TST_Status = CRYS_RSA_PKCS1v15_SignFinish(
		&TST_PrivContext,
		TST_Output,
		&TST_OutputSize);

	TST_ATP_CHECK_ERROR_RETURN(TST_Status,
		"CRYS_RSA_PKCS1v15_SignFinish",
		SEP_KeysOpRSA_TestData[TST_index].TST_Name,
		TST_MasterTestName);


	/* restore the encrypted key: */
	DX_VOS_FastMemCpy(TST_userKeyDataParam.encryptedKeyData_ptr, TST_savedUserEncryptedKeyBuffer, KMNG_RSA_ENCR_KEY_BUFF_SIZE_IN_BYTES);
	TST_userKeyDataParam.encryptedKeyDataLength = TST_savedUserEncryptedKeyDataLength;

	TST_KeySize  = sizeof(KMNG_RSA_WrappedKey_t);

	/* 7. Activate RSA key with type KMNG_RsaKeyOperationEncrypt */
	/*************************************************************/
	TST_Status = LLF_KMNG_ActivateRSAKey(&TST_masterKeyDataParam,
		&TST_userKeyDataParam,
		KMNG_RsaKeyOperationEncrypt,
		(DxUint8_t*)TST_RSAWrappedKey,
		&TST_KeySize);

	TST_ATP_CHECK_ERROR_RETURN(TST_Status,
		"LLF_KMNG_ActivateRSAKey",
		SEP_KeysOpRSA_TestData[TST_index].TST_Name,
		TST_MasterTestName);

	/* 8. Execute CRYS_KMNG_RSA_PKCS1v15_VerifyInit operation with activated key */
	/************************************************************/
	TST_Status = CRYS_KMNG_RSA_PKCS1v15_VerifyInit(TST_RSAWrappedKey,
		&TST_PubContext,
		SEP_KeysOpRSA_TestData[TST_index].TST_HASHMode);

	TST_ATP_CHECK_ERROR_RETURN(TST_Status,
		"CRYS_KMNG_RSA_PKCS1v15_VerifyInit",
		SEP_KeysOpRSA_TestData[TST_index].TST_Name,
		TST_MasterTestName);

	/* 9. Execute CRYS_KMNG_RSA_PKCS1v15_VerifyUpdate operation with activated key */
	/************************************************************/
	TST_Status = CRYS_RSA_PKCS1v15_VerifyUpdate(
		&TST_PubContext,
		SEP_KeysOpRSA_TestData[TST_index].TST_Data,
		SEP_KeysOpRSA_TestData[TST_index].TST_DataLen);

	TST_ATP_CHECK_ERROR_RETURN(TST_Status,
		"CRYS_RSA_PKCS1v15_VerifyUpdate",
		SEP_KeysOpRSA_TestData[TST_index].TST_Name,
		TST_MasterTestName);

	/* 10. Execute CRYS_KMNG_RSA_PKCS1v15_VerifyFinish operation with activated key */
	/************************************************************/
	TST_Status = CRYS_RSA_PKCS1v15_VerifyFinish(
		&TST_PubContext,
		TST_Output);

EXIT_ON_ERROR:
	return TST_Status; 

}

/************************ Public Functions ******************************/

/***********************************************************************
* Function Name:                                                           
*  SEP_KMNG_ATP_Ext_RSA_V15_ActivateOperation         
* 
* Inputs:
*
* Outputs:
*  CRYSError_t - Function error return                                     
*
* Description:                                                                 
*  This function Is a Acceptance test for RSA key activate function for V21 operations.
* Algorithm:       
* 1. Check RSA PKCS1V15 Encrypt,Decrypt operations with activated user key
* 2. Check RSA PKCS1V15 Sign,Verify integrated operations with activated user key
* 3. Check RSA PKCS1V15 Sign,Verify non integrated operations with activated user key
**************************************************************************************/                                
DxError_t SEP_KMNG_ATP_Ext_RSA_V15_ActivateOperation( void ) 
{
	/********* Variables Definitions ************/
	DxUint32_t	SEP_Error = DX_SUCCESS;

	ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"\n ******* SEP_KMNG_ATP_Ext_RSA_PKCS1V15_EncDec_ActivateOperation test ********* \n"));

	/* 1. Check RSA PKCS1V15 Encrypt,Decrypt operations with activated user key */
	/****************************************************************************/
	SEP_Error = SEP_KMNG_ATP_Ext_RSA_PKCS1V15_EncDec_ActivateOperation();

	SEP_Error = TST_CheckErrorReturn(SEP_Error,"SEP_KMNG_ATP_Ext_RSA_PKCS1V15_EncDec_ActivateOperation");

	ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"\n ******* SEP_KMNG_ATP_Ext_RSA_V15_SignVerify_Integrated_ActivateOperation test ********* \n"));

	/* 2. Check RSA PKCS1V15 Sign,Verify integrated operations with activated user key */
	/***********************************************************************************/
	SEP_Error = SEP_KMNG_ATP_Ext_RSA_V15_SignVerify_Integrated_ActivateOperation();

	SEP_Error = TST_CheckErrorReturn(SEP_Error,"SEP_KMNG_ATP_Ext_RSA_V15_SignVerify_Integrated_ActivateOperation");

	ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"\n ******* SEP_KMNG_ATP_Ext_RSA_V15_SignVerify_NonIntegrated_ActivateOperation test ********* \n"));

	/* 3. Check RSA PKCS1V15 Sign,Verify non integrated operations with activated user key */
	/***************************************************************************************/
	SEP_Error = SEP_KMNG_ATP_Ext_RSA_V15_SignVerify_NonIntegrated_ActivateOperation();

	SEP_Error = TST_CheckErrorReturn(SEP_Error,"SEP_KMNG_ATP_Ext_RSA_V15_SignVerify_NonIntegrated_ActivateOperation");

	return SEP_Error;
}



